import React from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  ScrollView,
  SafeAreaView,
} from 'react-native';
import R from '../../../assets/R';
import styles from './style';

const Filter3DayView = props => {
  const {
    currentDate,
    showMonthPicker,
    scrollViewRef,
    panResponder,
    getEventsForDate,
    getDayName,
    getMonthName,
    get3DaysDates,
    isToday,
    handleMonthSelect,
    calculateEventPosition,
  } = props;

  const renderMonthPicker = () => {
    if (!showMonthPicker) return null;

    return (
      <View style={styles.monthPickerContainer}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.monthPickerContent}>
          {Array.from({length: 12}, (_, index) => (
            <TouchableOpacity
              key={index}
              style={[
                styles.monthItem,
                currentDate.getMonth() === index && styles.monthItemSelected,
              ]}
              onPress={() => handleMonthSelect(index)}>
              <Text
                style={[
                  styles.monthItemText,
                  currentDate.getMonth() === index &&
                    styles.monthItemTextSelected,
                ]}>
                {getMonthName(index)}
              </Text>
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>
    );
  };

  const render3DaysHeader = () => {
    const threeDaysDates = get3DaysDates(currentDate);

    return (
      <View style={styles.daysHeaderContainer}>
        <View style={styles.timeColumnHeader} />
        {threeDaysDates.map((date, index) => {
          const isTodayDate = isToday(date);

          return (
            <View key={index} style={styles.dayHeaderCell}>
              <Text
                style={[
                  styles.dayHeaderText,
                  isTodayDate && styles.todayHeaderText,
                ]}>
                {getDayName(date)}
              </Text>
              <View
                style={[
                  styles.dayNumberContainer,
                  isTodayDate && styles.todayNumberContainer,
                ]}>
                <Text
                  style={[
                    styles.dayHeaderNumber,
                    isTodayDate && styles.todayHeaderNumber,
                  ]}>
                  {date.getDate()}
                </Text>
              </View>
            </View>
          );
        })}
      </View>
    );
  };

  const renderTimeSlots = () => {
    const hours = Array.from({length: 24}, (_, i) => i);
    const threeDaysDates = get3DaysDates(currentDate);

    return (
      <View style={styles.timeSlotsContainer} {...panResponder.panHandlers}>
        <ScrollView
          ref={scrollViewRef}
          showsVerticalScrollIndicator={false}
          contentContainerStyle={styles.scrollContent}>
          <View style={styles.timelineContainer}>
            <View style={styles.timeLabelsColumn}>
              {hours.map(hour => {
                const timeStr = hour.toString().padStart(2, '0') + ':00';
                return (
                  <View key={hour} style={styles.timeSlot}>
                    <Text style={styles.timeText}>{timeStr}</Text>
                  </View>
                );
              })}
            </View>

            <View style={styles.daysGridContainer}>
              {threeDaysDates.map((date, dayIndex) => (
                <View key={dayIndex} style={styles.dayColumn}>
                  {hours.map(hour => (
                    <View key={hour} style={styles.gridCell} />
                  ))}

                  {getEventsForDate(date).map(event => {
                    const {topPosition, height} = calculateEventPosition(
                      event.time,
                      event.endTime,
                    );

                    return (
                      <TouchableOpacity
                        key={event.id}
                        style={[
                          styles.eventCard,
                          {
                            position: 'absolute',
                            top: topPosition,
                            height: Math.max(height, 40),
                            left: 4,
                            right: 4,
                            zIndex: 10,
                            backgroundColor: R.colors.main,
                          },
                        ]}
                        activeOpacity={0.7}>
                        <Text
                          style={styles.eventTitle}
                          numberOfLines={height > 80 ? 3 : 2}>
                          {event.title}
                        </Text>
                        {height > 50 && (
                          <Text style={styles.eventSubtitle} numberOfLines={1}>
                            {event.subtitle}
                          </Text>
                        )}
                        <Text style={styles.eventTime}>
                          Thời gian học: {event.time} - {event.endTime},{' '}
                          {getDayName(date)} {date.getDate()}/
                          {date.getMonth() + 1}/{date.getFullYear()}
                        </Text>
                      </TouchableOpacity>
                    );
                  })}
                </View>
              ))}
            </View>
          </View>
        </ScrollView>
      </View>
    );
  };

  return (
    <SafeAreaView style={styles.container}>
      {renderMonthPicker()}
      {render3DaysHeader()}
      {renderTimeSlots()}
    </SafeAreaView>
  );
};

export default Filter3DayView;
