import React, {useState, useRef, useEffect} from 'react';
import {Dimensions, PanResponder} from 'react-native';
import {useNavigation, useFocusEffect} from '@react-navigation/native';
import {DeviceEventEmitter} from 'react-native';
import FilterWeekView from './view';

// ==================== HẰNG SỐ ====================
const {width: screenWidth, height: screenHeight} = Dimensions.get('window');
const HOUR_HEIGHT = 80;
const DAY_COLUMN_WIDTH = (screenWidth - 70) / 7;

const FilterWeek = ({navigation}) => {
  const navigationHook = useNavigation();

  // ==================== QUẢN LÝ STATE ====================
  // B1: State ngày tháng và lịch
  const [currentDate, setCurrentDate] = useState(new Date());
  const [selectedDate, setSelectedDate] = useState(new Date());

  // B2: State month picker
  const [showMonthPicker, setShowMonthPicker] = useState(false);

  // B3: Tham chiếu scroll
  const scrollViewRef = useRef(null);

  // ==================== EFFECTS ====================
  // Reset về ngày hiện tại khi chuyển màn hình
  useFocusEffect(
    React.useCallback(() => {
      const today = new Date();
      setCurrentDate(today);
      setSelectedDate(today);
      DeviceEventEmitter.emit('onDateChange', today);
      // Cập nhật header drawer với tháng hiện tại
      DeviceEventEmitter.emit('updateHeaderMonth', today.getMonth());
    }, [])
  );

  useEffect(() => {
    DeviceEventEmitter.emit('onDateChange', selectedDate);
  }, [selectedDate]);

  // ==================== HÀM TIỆN ÍCH ====================
  // T1: Định dạng ngày thành chuỗi
  const formatDateToString = date => {
    const year = date.getFullYear();
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const day = date.getDate().toString().padStart(2, '0');
    return `${year}-${month}-${day}`;
  };

  // T2: Kiểm tra ngày hôm nay
  const isToday = someDate => {
    const today = new Date();
    return (
      someDate.getDate() === today.getDate() &&
      someDate.getMonth() === today.getMonth() &&
      someDate.getFullYear() === today.getFullYear()
    );
  };

  // T3: Lấy tên ngày trong tuần
  const getDayName = date => {
    const days = ['CN', 'T2', 'T3', 'T4', 'T5', 'T6', 'T7'];
    return days[date.getDay()];
  };

  // T4: Lấy tên tháng
  const getMonthName = monthIndex => {
    const months = [
      'Tháng 1',
      'Tháng 2',
      'Tháng 3',
      'Tháng 4',
      'Tháng 5',
      'Tháng 6',
      'Tháng 7',
      'Tháng 8',
      'Tháng 9',
      'Tháng 10',
      'Tháng 11',
      'Tháng 12',
    ];
    return months[monthIndex];
  };

  // T5: Lấy các ngày trong tuần
  const getWeekDates = date => {
    const startOfWeek = new Date(date);
    startOfWeek.setDate(date.getDate() - date.getDay());

    const weekDates = [];
    for (let i = 0; i < 7; i++) {
      const weekDate = new Date(startOfWeek);
      weekDate.setDate(startOfWeek.getDate() + i);
      weekDates.push(weekDate);
    }
    return weekDates;
  };

  // T6: Tính toán vị trí sự kiện
  const calculateEventPosition = (startTime, endTime) => {
    const [startHour, startMinute] = startTime.split(':').map(Number);
    const [endHour, endMinute] = endTime.split(':').map(Number);

    const startTotalMinutes = startHour * 60 + startMinute;
    const endTotalMinutes = endHour * 60 + endMinute;
    const durationMinutes = endTotalMinutes - startTotalMinutes;

    const topPosition = (startTotalMinutes / 60) * HOUR_HEIGHT;
    const height = (durationMinutes / 60) * HOUR_HEIGHT;

    return {topPosition, height};
  };

  // ==================== QUẢN LÝ DỮ LIỆU ====================
  // D1: Tạo dữ liệu sự kiện mẫu
  const createMockEvents = () => {
    return [
      {
        id: '1',
        title: 'Lịch vào trực lớp TTCĐT 445.T1',
        subtitle: 'CS Địa lý 4D',
        time: '07:00',
        endTime: '09:00',
        date: '2025-09-26',
        type: 'class',
      },
      {
        id: '2',
        title: 'Meeting team development',
        subtitle: 'Phòng họp A1',
        time: '10:00',
        endTime: '11:30',
        date: '2025-09-26',
        type: 'meeting',
      },
      {
        id: '3',
        title: 'Training React Native',
        subtitle: 'Online Zoom',
        time: '14:00',
        endTime: '16:00',
        date: '2025-09-26',
        type: 'training',
      },
    ];
  };

  // D2: Xử lý dữ liệu sự kiện
  const mockEvents = createMockEvents();

  // D3: Hàm truy vấn sự kiện
  const getEventsForDate = date => {
    const dateStr = formatDateToString(date);
    return mockEvents.filter(event => event.date === dateStr);
  };

  // ==================== HỆ THỐNG ANIMATION ====================
  // A1: Thiết lập PanResponder
  const panResponder = PanResponder.create({
    onMoveShouldSetPanResponder: (evt, gestureState) => {
      return Math.abs(gestureState.dx) > 30 && Math.abs(gestureState.dy) < 100;
    },
    onPanResponderMove: (evt, gestureState) => {},
    onPanResponderRelease: (evt, gestureState) => {
      if (gestureState.dx > 50) {
        swipeToPrevWeek();
      } else if (gestureState.dx < -50) {
        swipeToNextWeek();
      }
    },
  });

  // ==================== XỬ LÝ SỰ KIỆN ====================
  // X1: Xử lý chọn tháng
  const handleMonthSelect = monthIndex => {
    const newDate = new Date(currentDate);
    newDate.setMonth(monthIndex);
    setCurrentDate(newDate);
    setSelectedDate(newDate);
    setShowMonthPicker(false);
    DeviceEventEmitter.emit('onDateChange', newDate);
    // Cập nhật header drawer với tháng mới
    DeviceEventEmitter.emit('updateHeaderMonth', monthIndex);
  };

  // X2: Xử lý chuyển tuần
  const swipeToNextWeek = () => {
    const nextWeek = new Date(currentDate);
    nextWeek.setDate(currentDate.getDate() + 7);
    setCurrentDate(nextWeek);
    setSelectedDate(nextWeek);
    DeviceEventEmitter.emit('onDateChange', nextWeek);
    // Cập nhật header drawer nếu tháng thay đổi
    DeviceEventEmitter.emit('updateHeaderMonth', nextWeek.getMonth());
  };

  const swipeToPrevWeek = () => {
    const prevWeek = new Date(currentDate);
    prevWeek.setDate(currentDate.getDate() - 7);
    setCurrentDate(prevWeek);
    setSelectedDate(prevWeek);
    DeviceEventEmitter.emit('onDateChange', prevWeek);
    // Cập nhật header drawer nếu tháng thay đổi
    DeviceEventEmitter.emit('updateHeaderMonth', prevWeek.getMonth());
  };

  // X3: Xử lý toggle month picker
  const toggleMonthPicker = () => {
    setShowMonthPicker(!showMonthPicker);
  };

  return (
    <FilterWeekView
      navigation={navigation}
      currentDate={currentDate}
      selectedDate={selectedDate}
      showMonthPicker={showMonthPicker}
      scrollViewRef={scrollViewRef}
      panResponder={panResponder}
      isToday={isToday}
      getDayName={getDayName}
      getMonthName={getMonthName}
      getWeekDates={getWeekDates}
      getEventsForDate={getEventsForDate}
      calculateEventPosition={calculateEventPosition}
      handleMonthSelect={handleMonthSelect}
      toggleMonthPicker={toggleMonthPicker}
      HOUR_HEIGHT={HOUR_HEIGHT}
      DAY_COLUMN_WIDTH={DAY_COLUMN_WIDTH}
    />
  );
};

export default FilterWeek;
