import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  ScrollView,
  StyleSheet,
  Dimensions,
  SafeAreaView,
  DeviceEventEmitter,
  PanResponder,
} from 'react-native';
import R from '../../../assets/R';

const { width: screenWidth, height: screenHeight } = Dimensions.get('window');
const HOUR_HEIGHT = 80;
const DAY_COLUMN_WIDTH = (screenWidth - 70) / 7;

const FilterWeekView = ({ navigation }) => {
  const [currentDate, setCurrentDate] = useState(new Date());
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [showMonthPicker, setShowMonthPicker] = useState(false);
  const scrollViewRef = useRef(null);

  useEffect(() => {
    DeviceEventEmitter.emit('onDateChange', selectedDate);
  }, [selectedDate]);

  const createMockEvents = () => {
    return [
      {
        id: '1',
        title: 'Lịch vào trực lớp TTCĐT 445.T1',
        subtitle: 'CS Địa lý 4D',
        time: '07:00',
        endTime: '09:00',
        date: '2025-07-21',
        type: 'class',
      },
      {
        id: '2',
        title: 'Meeting team development',
        subtitle: 'Phòng họp A1',
        time: '10:00',
        endTime: '11:30',
        date: '2025-07-22',
        type: 'meeting',
      },
      {
        id: '3',
        title: 'Training React Native',
        subtitle: 'Online Zoom',
        time: '14:00',
        endTime: '16:00',
        date: '2025-07-23',
        type: 'training',
      },
      {
        id: '4',
        title: 'Code Review Session',
        subtitle: 'Dev Team',
        time: '09:00',
        endTime: '10:30',
        date: '2025-07-24',
        type: 'review',
      },
      {
        id: '5',
        title: 'Sprint Planning',
        subtitle: 'Scrum Team',
        time: '15:00',
        endTime: '17:00',
        date: '2025-07-25',
        type: 'meeting',
      },
      {
        id: '6',
        title: 'Tech Talk',
        subtitle: 'Conference Room',
        time: '11:00',
        endTime: '12:00',
        date: '2025-07-26',
        type: 'training',
      },
    ];
  };

  const mockEvents = createMockEvents();

  const formatDateToString = (date) => {
    const year = date.getFullYear();
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const day = date.getDate().toString().padStart(2, '0');
    return `${year}-${month}-${day}`;
  };

  const getEventsForDate = (date) => {
    const dateStr = formatDateToString(date);
    return mockEvents.filter(event => event.date === dateStr);
  };

  const getDayName = (date) => {
    const days = ['CN', 'T2', 'T3', 'T4', 'T5', 'T6', 'T7'];
    return days[date.getDay()];
  };

  const getMonthName = (monthIndex) => {
    const months = [
      'Tháng 1', 'Tháng 2', 'Tháng 3', 'Tháng 4', 'Tháng 5', 'Tháng 6',
      'Tháng 7', 'Tháng 8', 'Tháng 9', 'Tháng 10', 'Tháng 11', 'Tháng 12',
    ];
    return months[monthIndex];
  };

  const getWeekDates = (date) => {
    const startOfWeek = new Date(date);
    startOfWeek.setDate(date.getDate() - date.getDay());

    const weekDates = [];
    for (let i = 0; i < 7; i++) {
      const weekDate = new Date(startOfWeek);
      weekDate.setDate(startOfWeek.getDate() + i);
      weekDates.push(weekDate);
    }
    return weekDates;
  };

  const handleMonthSelect = (monthIndex) => {
    const newDate = new Date(currentDate);
    newDate.setMonth(monthIndex);
    setCurrentDate(newDate);
    setSelectedDate(newDate);
    setShowMonthPicker(false);
    DeviceEventEmitter.emit('onDateChange', newDate);
  };

  const swipeToNextWeek = () => {
    const nextWeek = new Date(currentDate);
    nextWeek.setDate(currentDate.getDate() + 7);
    setCurrentDate(nextWeek);
    setSelectedDate(nextWeek);
    DeviceEventEmitter.emit('onDateChange', nextWeek);
  };

  const swipeToPrevWeek = () => {
    const prevWeek = new Date(currentDate);
    prevWeek.setDate(currentDate.getDate() - 7);
    setCurrentDate(prevWeek);
    setSelectedDate(prevWeek);
    DeviceEventEmitter.emit('onDateChange', prevWeek);
  };

  const toggleMonthPicker = () => {
    setShowMonthPicker(!showMonthPicker);
  };

  const panResponder = PanResponder.create({
    onMoveShouldSetPanResponder: (evt, gestureState) => {
      return Math.abs(gestureState.dx) > 30 && Math.abs(gestureState.dy) < 100;
    },
    onPanResponderMove: (evt, gestureState) => { },
    onPanResponderRelease: (evt, gestureState) => {
      if (gestureState.dx > 50) {
        swipeToPrevWeek();
      } else if (gestureState.dx < -50) {
        swipeToNextWeek();
      }
    },
  });

  const calculateEventPosition = (startTime, endTime) => {
    const [startHour, startMinute] = startTime.split(':').map(Number);
    const [endHour, endMinute] = endTime.split(':').map(Number);

    const startTotalMinutes = startHour * 60 + startMinute;
    const endTotalMinutes = endHour * 60 + endMinute;
    const durationMinutes = endTotalMinutes - startTotalMinutes;

    const topPosition = (startTotalMinutes / 60) * HOUR_HEIGHT;
    const height = (durationMinutes / 60) * HOUR_HEIGHT;

    return { topPosition, height };
  };

  const renderMonthPicker = () => {
    if (!showMonthPicker) return null;

    return (
      <View style={styles.monthPickerContainer}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.monthPickerContent}>
          {Array.from({ length: 12 }, (_, index) => (
            <TouchableOpacity
              key={index}
              style={[
                styles.monthItem,
                currentDate.getMonth() === index && styles.monthItemSelected,
              ]}
              onPress={() => handleMonthSelect(index)}>
              <Text
                style={[
                  styles.monthItemText,
                  currentDate.getMonth() === index && styles.monthItemTextSelected,
                ]}>
                {getMonthName(index)}
              </Text>
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>
    );
  };

  const renderWeekHeader = () => {
    const weekDates = getWeekDates(currentDate);

    return (
      <View style={styles.weekHeaderContainer}>
        <View style={styles.timeColumnHeader} />
        {weekDates.map((date, index) => (
          <View key={index} style={styles.dayHeaderCell}>
            <Text style={styles.dayHeaderText}>{getDayName(date)}</Text>
            <Text style={styles.dayHeaderNumber}>{date.getDate()}</Text>
          </View>
        ))}
      </View>
    );
  };

  const renderTimeSlots = () => {
    const hours = Array.from({ length: 24 }, (_, i) => i);
    const weekDates = getWeekDates(currentDate);

    return (
      <View style={styles.timeSlotsContainer} {...panResponder.panHandlers}>
        <ScrollView
          ref={scrollViewRef}
          showsVerticalScrollIndicator={false}
          contentContainerStyle={styles.scrollContent}>

          <View style={styles.timelineContainer}>
            <View style={styles.timeLabelsColumn}>
              {hours.map((hour) => {
                const timeStr = hour.toString().padStart(2, '0') + ':00';
                return (
                  <View key={hour} style={styles.timeSlot}>
                    <Text style={styles.timeText}>{timeStr}</Text>
                  </View>
                );
              })}
            </View>

            <View style={styles.weekGridContainer}>
              {weekDates.map((date, dayIndex) => (
                <View key={dayIndex} style={styles.dayColumn}>
                  {hours.map((hour) => (
                    <View key={hour} style={styles.gridCell} />
                  ))}

                  {getEventsForDate(date).map((event) => {
                    const { topPosition, height } = calculateEventPosition(event.time, event.endTime);

                    return (
                      <TouchableOpacity
                        key={event.id}
                        style={[
                          styles.eventCard,
                          {
                            position: 'absolute',
                            top: topPosition,
                            height: Math.max(height, 40),
                            left: 2,
                            right: 2,
                            zIndex: 10,
                            backgroundColor: R.colors.blue500,
                          }
                        ]}
                        activeOpacity={0.7}>
                        <Text style={styles.eventTitle} numberOfLines={height > 60 ? 2 : 1}>
                          {event.title}
                        </Text>
                        {height > 40 && (
                          <Text style={styles.eventSubtitle} numberOfLines={1}>
                            {event.subtitle}
                          </Text>
                        )}
                        <Text style={styles.eventTime}>
                          {event.time} - {event.endTime}
                        </Text>
                      </TouchableOpacity>
                    );
                  })}
                </View>
              ))}
            </View>
          </View>
        </ScrollView>
      </View>
    );
  };

  return (
    <SafeAreaView style={styles.container}>
      {renderMonthPicker()}
      {renderWeekHeader()}
      {renderTimeSlots()}
    </SafeAreaView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: R.colors.white,
  },
  monthPickerContainer: {
    backgroundColor: R.colors.white,
    borderBottomWidth: 1,
    borderBottomColor: R.colors.grey_200,
    paddingVertical: 10,
  },
  monthPickerContent: {
    paddingHorizontal: 15,
  },
  monthItem: {
    paddingHorizontal: 20,
    paddingVertical: 8,
    marginRight: 10,
    borderRadius: 20,
    backgroundColor: R.colors.grey_50,
  },
  monthItemSelected: {
    backgroundColor: R.colors.blue500,
  },
  monthItemText: {
    fontSize: R.fontsize.fontsSize12,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.black,
  },
  monthItemTextSelected: {
    color: R.colors.white,
    fontFamily: R.fonts.InterMedium,
  },
  weekHeaderContainer: {
    flexDirection: 'row',
    backgroundColor: R.colors.grey_200,
    borderBottomWidth: 1,
    borderBottomColor: R.colors.grey_200,
  },
  timeColumnHeader: {
    width: 70,
  },
  dayHeaderCell: {
    width: DAY_COLUMN_WIDTH,
    alignItems: 'center',
    justifyContent: 'center',
    borderRightWidth: 1,
    borderRightColor: R.colors.grey_200,
  },
  dayHeaderText: {
    fontSize: R.fontsize.fontsSize10,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.grey_600,
  },
  dayHeaderNumber: {
    fontSize: R.fontsize.fontsSize12,
    fontFamily: R.fonts.InterMedium,
    color: R.colors.black,
  },
  timeSlotsContainer: {
    flex: 1,
    backgroundColor: R.colors.white,
  },
  timelineContainer: {
    flexDirection: 'row',
    position: 'relative',
  },
  timeLabelsColumn: {
    width: 70,
    borderRightWidth: 1,
    borderRightColor: R.colors.grey_200,
  },
  weekGridContainer: {
    flex: 1,
    flexDirection: 'row',
  },
  dayColumn: {
    width: DAY_COLUMN_WIDTH,
    position: 'relative',
    borderRightWidth: 1,
    borderRightColor: R.colors.grey_200,
  },
  timeSlot: {
    height: HOUR_HEIGHT,
    alignItems: 'center',
    justifyContent: 'center',
    borderBottomWidth: 1,
    borderBottomColor: R.colors.grey_200,
  },
  gridCell: {
    height: HOUR_HEIGHT,
    borderBottomWidth: 1,
    borderBottomColor: R.colors.grey_200,
    width: '100%',
  },
  timeText: {
    fontSize: R.fontsize.fontSizeContent,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.black,
  },
  eventCard: {
    borderRadius: 10,
    paddingHorizontal: 4,
    paddingVertical: 2,
    justifyContent: 'center',
  },
  eventTitle: {
    fontSize: R.fontsize.fontsSize10,
    fontFamily: R.fonts.InterMedium,
    color: R.colors.white,
    marginBottom: 1,
  },
  eventSubtitle: {
    fontSize: R.fontsize.fontsSize8,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.white,
  },
  eventTime: {
    fontSize: R.fontsize.fontsSize8,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.white,
  },
});

export default FilterWeekView;