import React, {useState, useMemo, useCallback} from 'react';
import {
  DeviceEventEmitter,
  View,
  Text,
  Image,
  TouchableOpacity,
  StyleSheet,
} from 'react-native';
import {
  DrawerContentScrollView,
  useDrawerProgress,
} from '@react-navigation/drawer';
import R from '../../assets/R';
import * as ScreenName from '../ScreenNames';
import CheckBox from '../../components/CheckBox';

const IcDate = R.images.icDate;
const Ic3Date = R.images.ic3Date;
const IcWeek = R.images.icWeek;
const IcMonth = R.images.icMonth;

const CustomDrawerContent = props => {
  const progress = useDrawerProgress();
  const {navigation, state} = props;

  const checkboxConfigs = useMemo(
    () => [
      {key: 'departmentCalendar', label: 'Lịch phòng ban'},
      {key: 'personalCalendar', label: 'Lịch cá nhân'},
      {key: 'teachingCalendar', label: 'Lịch giảng dạy'},
      {key: 'examCalendar', label: 'Lịch coi thi'},
      {key: 'examRoomCalendar', label: 'Lịch sử dụng phòng thi'},
    ],
    [],
  );

  const filterConfigs = useMemo(
    () => [
      {
        label: 'Theo ngày',
        screenName: ScreenName.FILTERDATE,
        icon: <IcDate stroke={'black'} />,
      },
      {
        label: '3 ngày',
        screenName: ScreenName.FILTER3DATE,
        icon: <Ic3Date />,
      },
      {
        label: 'Tuần',
        screenName: ScreenName.FILTERWEEK,
        icon: <IcWeek />,
      },
      {
        label: 'Tháng',
        screenName: ScreenName.FILTERMONTH,
        icon: <IcMonth />,
      },
    ],
    [],
  );

  const [checkboxStates, setCheckboxStates] = useState({
    departmentCalendar: false,
    personalCalendar: false,
    teachingCalendar: false,
    examCalendar: false,
    examRoomCalendar: false,
  });

  const currentRouteName = state.routes[state.index]?.name;

  const handleCheckboxChange = useCallback((key, newValue) => {
    setCheckboxStates(prev => ({
      ...prev,
      [key]: newValue,
    }));

    DeviceEventEmitter.emit('onCheckboxChange', {key, value: newValue});
  }, []);

  const handleScreenNavigation = useCallback(
    screenName => {
      navigation.navigate(screenName);
      navigation.closeDrawer();
    },
    [navigation],
  );

  const renderCheckbox = useCallback(
    config => (
      <CheckBox
        key={config.key}
        value={checkboxStates[config.key]}
        onValueChange={newValue => handleCheckboxChange(config.key, newValue)}
        label={config.label}
        labelSpacing={30}
        size={20}
        labelSize={R.fontsize.fontsSize12}
        checkedColor={R.colors.black}
        marginBottom={20}
        labelFontFamily={R.fonts.InterRegular}
        labelWeight={400}
      />
    ),
    [checkboxStates, handleCheckboxChange],
  );

  const renderFilterItem = useCallback(
    item => {
      const isSelected = currentRouteName === item.screenName;

      return (
        <TouchableOpacity
          key={item.screenName}
          style={[styles.filterItem, isSelected && styles.filterItemSelected]}
          onPress={() => handleScreenNavigation(item.screenName)}
          activeOpacity={0.7}>
          {item.icon}

          <Text
            style={[
              styles.filterLabel,
              isSelected && styles.filterLabelSelected,
            ]}>
            {item.label}
          </Text>
        </TouchableOpacity>
      );
    },
    [currentRouteName, handleScreenNavigation],
  );

  return (
    <DrawerContentScrollView
      {...props}
      contentContainerStyle={styles.scrollContainer}>
      <View style={styles.logoContainer}>
        <Image
          source={R.images.igLogo}
          style={styles.logo}
          resizeMode="contain"
        />
      </View>

      <View style={styles.filterSection}>
        {filterConfigs.map(renderFilterItem)}
      </View>

      <View style={styles.divider} />

      <View style={styles.checkboxSection}>
        <Text style={styles.sectionTitle}>Loại lịch</Text>
        {checkboxConfigs.map(renderCheckbox)}
      </View>
    </DrawerContentScrollView>
  );
};

const styles = StyleSheet.create({
  scrollContainer: {
    flexGrow: 1,
    paddingBottom: 20,
  },
  logoContainer: {
    alignItems: 'center',
    marginVertical: 20,
  },
  logo: {
    width: 149,
    height: 24,
  },
  filterSection: {
    paddingHorizontal: 0,
  },
  filterItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 15,
    marginVertical: 4,
  },
  filterIcon: {
    width: 20,
    height: 20,
  },
  filterItemSelected: {
    backgroundColor: R.colors.main,
    borderRadius: 50,
    marginHorizontal: 15,
  },
  filterLabel: {
    marginLeft: 30,
    fontSize: R.fontsize.fontsSize12,
    color: R.colors.black,
    flex: 1,
    fontFamily: R.fonts.InterMedium,
    fontWeight: '600',
  },
  filterLabelSelected: {
    color: R.colors.white,
    fontWeight: '500',
    fontFamily: R.fonts.InterSemiBold,
  },
  divider: {
    height: 1,
    backgroundColor: R.colors.black,
    marginHorizontal: 15,
    marginTop: 21,
  },
  checkboxSection: {
    paddingHorizontal: 15,
    marginTop: 24,
  },
  sectionTitle: {
    fontWeight: '600',
    fontFamily: R.fonts.InterMedium,
    marginBottom: 10,
    fontSize: R.fontsize.fontsSize16,
    color: R.colors.black,
  },
});

export default CustomDrawerContent;
