import React, {useState, useRef, useEffect} from 'react';
import {Dimensions, PanResponder} from 'react-native';
import {useNavigation, useFocusEffect} from '@react-navigation/native';
import {DeviceEventEmitter} from 'react-native';
import Filter3DayView from './view';

// ==================== HẰNG SỐ ====================
const {width: screenWidth, height: screenHeight} = Dimensions.get('window');
const HOUR_HEIGHT = 80;
const DAY_COLUMN_WIDTH = (screenWidth - 70) / 3;

const Filter3Day = ({navigation}) => {
  const navigationHook = useNavigation();

  // ==================== QUẢN LÝ STATE ====================
  const [currentDate, setCurrentDate] = useState(new Date());
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [showMonthPicker, setShowMonthPicker] = useState(false);
  const scrollViewRef = useRef(null);

  // ==================== EFFECTS ====================
  // Reset về ngày hiện tại khi chuyển màn hình
  useFocusEffect(
    React.useCallback(() => {
      const today = new Date();
      setCurrentDate(today);
      setSelectedDate(today);
      DeviceEventEmitter.emit('onDateChange', today);
      // Cập nhật header drawer với tháng hiện tại
      DeviceEventEmitter.emit('updateHeaderMonth', today.getMonth());
    }, [])
  );

  useEffect(() => {
    DeviceEventEmitter.emit('onDateChange', selectedDate);
  }, [selectedDate]);

  // ==================== HÀM TIỆN ÍCH ====================
  // T1: Định dạng ngày thành chuỗi
  const formatDateToString = date => {
    const year = date.getFullYear();
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const day = date.getDate().toString().padStart(2, '0');
    return `${year}-${month}-${day}`;
  };

  // T2: Kiểm tra ngày hôm nay
  const isToday = someDate => {
    const today = new Date();
    return (
      someDate.getDate() === today.getDate() &&
      someDate.getMonth() === today.getMonth() &&
      someDate.getFullYear() === today.getFullYear()
    );
  };

  // T3: Lấy tên ngày trong tuần
  const getDayName = date => {
    const days = ['CN', 'T2', 'T3', 'T4', 'T5', 'T6', 'T7'];
    return days[date.getDay()];
  };

  // T4: Lấy tên tháng
  const getMonthName = monthIndex => {
    const months = [
      'Tháng 1',
      'Tháng 2',
      'Tháng 3',
      'Tháng 4',
      'Tháng 5',
      'Tháng 6',
      'Tháng 7',
      'Tháng 8',
      'Tháng 9',
      'Tháng 10',
      'Tháng 11',
      'Tháng 12',
    ];
    return months[monthIndex];
  };

  // T5: Lấy 3 ngày liên tiếp
  const get3DaysDates = date => {
    const threeDayDates = [];
    for (let i = -1; i <= 1; i++) {
      const dayDate = new Date(date);
      dayDate.setDate(date.getDate() + i);
      threeDayDates.push(dayDate);
    }
    return threeDayDates;
  };

  // T6: Tính toán vị trí sự kiện
  const calculateEventPosition = (startTime, endTime) => {
    const [startHour, startMinute] = startTime.split(':').map(Number);
    const [endHour, endMinute] = endTime.split(':').map(Number);

    const startTotalMinutes = startHour * 60 + startMinute;
    const endTotalMinutes = endHour * 60 + endMinute;
    const durationMinutes = endTotalMinutes - startTotalMinutes;

    const topPosition = (startTotalMinutes / 60) * HOUR_HEIGHT;
    const height = (durationMinutes / 60) * HOUR_HEIGHT;

    return {topPosition, height};
  };

  // ==================== QUẢN LÝ DỮ LIỆU ====================
  // D1: Tạo dữ liệu sự kiện mẫu
  const createMockEvents = () => {
    const today = new Date();
    const todayStr = formatDateToString(today);
    
    const tomorrow = new Date(today);
    tomorrow.setDate(today.getDate() + 1);
    const tomorrowStr = formatDateToString(tomorrow);
    
    const yesterday = new Date(today);
    yesterday.setDate(today.getDate() - 1);
    const yesterdayStr = formatDateToString(yesterday);

    return [
      // Sự kiện hôm qua
      {
        id: '1',
        title: 'Họp nội bộ khoa',
        subtitle: 'Phòng họp A1',
        time: '09:00',
        endTime: '11:00',
        date: yesterdayStr,
        type: 'meeting',
      },
      // Sự kiện hôm nay
      {
        id: '2',
        title: 'Lịch vào trực lớp TTCĐT 445.T1',
        subtitle: 'CS Địa lý 4D',
        time: '07:00',
        endTime: '09:00',
        date: todayStr,
        type: 'class',
      },
      {
        id: '3',
        title: 'Training React Native',
        subtitle: 'Online Zoom',
        time: '14:00',
        endTime: '16:00',
        date: todayStr,
        type: 'training',
      },
      // Sự kiện ngày mai
      {
        id: '4',
        title: 'Workshop AI trong giáo dục',
        subtitle: 'Hội trường lớn',
        time: '13:00',
        endTime: '17:00',
        date: tomorrowStr,
        type: 'workshop',
      },
    ];
  };

  // D2: Xử lý dữ liệu sự kiện
  const mockEvents = createMockEvents();

  // D3: Hàm truy vấn sự kiện
  const getEventsForDate = date => {
    const dateStr = formatDateToString(date);
    return mockEvents.filter(event => event.date === dateStr);
  };

  // ==================== HỆ THỐNG ANIMATION ====================
  // A1: Thiết lập PanResponder
  const panResponder = PanResponder.create({
    onMoveShouldSetPanResponder: (evt, gestureState) => {
      return Math.abs(gestureState.dx) > 30 && Math.abs(gestureState.dy) < 100;
    },
    onPanResponderMove: (evt, gestureState) => {},
    onPanResponderRelease: (evt, gestureState) => {
      if (gestureState.dx > 50) {
        swipeToPrevThreeDay();
      } else if (gestureState.dx < -50) {
        swipeToNextThreeDay();
      }
    },
  });

  // ==================== XỬ LÝ SỰ KIỆN ====================
  // X1: Xử lý chọn tháng
  const handleMonthSelect = monthIndex => {
    const newDate = new Date(currentDate);
    newDate.setMonth(monthIndex);
    setCurrentDate(newDate);
    setSelectedDate(newDate);
    setShowMonthPicker(false);
    DeviceEventEmitter.emit('onDateChange', newDate);
    // Cập nhật header drawer với tháng mới
    DeviceEventEmitter.emit('updateHeaderMonth', monthIndex);
  };

  // X2: Xử lý chuyển 3 ngày
  const swipeToNextThreeDay = () => {
    const nextThreeDay = new Date(currentDate);
    nextThreeDay.setDate(currentDate.getDate() + 3);
    setCurrentDate(nextThreeDay);
    setSelectedDate(nextThreeDay);
    DeviceEventEmitter.emit('onDateChange', nextThreeDay);
    // Cập nhật header drawer nếu tháng thay đổi
    DeviceEventEmitter.emit('updateHeaderMonth', nextThreeDay.getMonth());
  };

  const swipeToPrevThreeDay = () => {
    const prevThreeDay = new Date(currentDate);
    prevThreeDay.setDate(currentDate.getDate() - 3);
    setCurrentDate(prevThreeDay);
    setSelectedDate(prevThreeDay);
    DeviceEventEmitter.emit('onDateChange', prevThreeDay);
    // Cập nhật header drawer nếu tháng thay đổi
    DeviceEventEmitter.emit('updateHeaderMonth', prevThreeDay.getMonth());
  };

  // X3: Xử lý toggle month picker
  const toggleMonthPicker = () => {
    setShowMonthPicker(!showMonthPicker);
  };

  return (
    <Filter3DayView
      navigation={navigation}
      currentDate={currentDate}
      selectedDate={selectedDate}
      showMonthPicker={showMonthPicker}
      scrollViewRef={scrollViewRef}
      panResponder={panResponder}
      isToday={isToday}
      getDayName={getDayName}
      getMonthName={getMonthName}
      get3DaysDates={get3DaysDates}
      getEventsForDate={getEventsForDate}
      calculateEventPosition={calculateEventPosition}
      handleMonthSelect={handleMonthSelect}
      toggleMonthPicker={toggleMonthPicker}
      HOUR_HEIGHT={HOUR_HEIGHT}
      DAY_COLUMN_WIDTH={DAY_COLUMN_WIDTH}
    />
  );
};

export default Filter3Day;
