import React from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  ScrollView,
  StyleSheet,
  Dimensions,
  SafeAreaView,
} from 'react-native';
import R from '../../../assets/R';

const {width: screenWidth, height: screenHeight} = Dimensions.get('window');
const HOUR_HEIGHT = 80;

const FilterDateView= ({
  navigation,
  currentDate,
  selectedDate,
  showMonthPicker,
  scrollViewRef,
  panResponder,
  getEventsForDate,
  getDayName,
  getMonthName,
  handleMonthSelect,
  toggleMonthPicker,
  calculateEventPosition,
}) => {
  const renderMonthPicker = () => {
    if (!showMonthPicker) return null;

    return (
      <View style={styles.monthPickerContainer}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.monthPickerContent}>
          {Array.from({length: 12}, (_, index) => (
            <TouchableOpacity
              key={index}
              style={[
                styles.monthItem,
                currentDate.getMonth() === index && styles.monthItemSelected,
              ]}
              onPress={() => handleMonthSelect(index)}>
              <Text
                style={[
                  styles.monthItemText,
                  currentDate.getMonth() === index && styles.monthItemTextSelected,
                ]}>
                {getMonthName(index)}
              </Text>
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>
    );
  };

  const renderDateInfo = () => {
    return (
      <View style={{backgroundColor: R.colors.grey_200}}>
             <View style={styles.dateInfoContainer}>
        <Text style={styles.dayName}>{getDayName(selectedDate)}</Text>
        <Text style={styles.dayNumber}>{selectedDate.getDate()}</Text>
      </View>
      </View>
   
    );
  };

  const renderTimeSlots = () => {
    const hours = Array.from({length: 24}, (_, i) => i);
    const selectedEvents = getEventsForDate(selectedDate);

    return (
      <View style={styles.timeSlotsContainer} {...panResponder.panHandlers}>
        <ScrollView
          ref={scrollViewRef}
          showsVerticalScrollIndicator={false}
          contentContainerStyle={styles.scrollContent}>
          
          <View style={styles.timelineContainer}>
            <View style={styles.timeLabelsColumn}>
              {hours.map((hour) => {
                const timeStr = hour.toString().padStart(2, '0') + ':00';
                return (
                  <View key={hour} style={styles.timeSlot}>
                    <Text style={styles.timeText}>{timeStr}</Text>
                  </View>
                );
              })}
            </View>

            <View style={styles.eventsColumn}>
              {hours.map((hour) => (
                <View key={hour} style={styles.gridLine} />
              ))}

              {selectedEvents.map((event) => {
                const { topPosition, height } = calculateEventPosition(event.time, event.endTime);
                
                return (
                  <TouchableOpacity
                    key={event.id}
                    style={[
                      styles.eventCard,
                      {
                        position: 'absolute',
                        top: topPosition,
                        height: Math.max(height, 40), 
                        left: 5,
                        right: 15,
                        zIndex: 10,
                        backgroundColor: R.colors.main,
                      }
                    ]}
                    activeOpacity={0.7}>
                    <Text style={styles.eventTitle} numberOfLines={height > 60 ? 2 : 1}>
                      {event.title}
                    </Text>
                    {height > 40 && (
                      <Text style={styles.eventSubtitle} numberOfLines={1}>
                        {event.subtitle}
                      </Text>
                    )}
                    <Text style={styles.eventTime}>
                      {event.time} - {event.endTime}
                    </Text>
                  </TouchableOpacity>
                );
              })}
            </View>
          </View>
        </ScrollView>
      </View>
    );
  };

  return (
    <SafeAreaView style={styles.container}>
      {renderMonthPicker()}
      {renderDateInfo()}
      {renderTimeSlots()}
    </SafeAreaView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: R.colors.white,
  },
  monthPickerContainer: {
    backgroundColor: R.colors.white,
    borderBottomWidth: 1,
    borderBottomColor: R.colors.grey_200,
    paddingVertical: 10,
  },
  monthPickerContent: {
    paddingHorizontal: 15,
  },
  monthItem: {
    paddingHorizontal: 20,
    paddingVertical: 8,
    marginRight: 10,
    borderRadius: 20,
    backgroundColor: R.colors.grey_50,
  },
  dateInfoContainer: {
    paddingHorizontal: 15,
    paddingVertical: 12,
    alignItems: 'center',
    justifyContent:'center',
    maxWidth:70,
    borderRightWidth:1,
    borderRightColor:R.colors.grey_200,
  },
  dayName: {
    fontSize: R.fontsize.fontsSize12,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.black,
    marginBottom: 2,
  },
  dayNumber: {
    fontSize: R.fontsize.fontsSize14,
    fontFamily: R.fonts.InterSemiBold,
    color: R.colors.blue500,
  },
  timeSlotsContainer: {
    flex: 1,
    backgroundColor: R.colors.white,
  },
  scrollContent: {
    paddingBottom: 50,
  },
  timelineContainer: {
    flexDirection: 'row',
    position: 'relative',
  },
  timeLabelsColumn: {
    minWidth: 70,
    borderRightWidth: 1,
    borderRightColor: R.colors.grey_200,
  },
  eventsColumn: {
    flex: 1,
    position: 'relative',
    minHeight: 24 * HOUR_HEIGHT,
  },
  timeSlot: {
    height: HOUR_HEIGHT,
    alignItems: 'center',
    justifyContent: 'center',
    borderBottomWidth: 1,
    borderBottomColor: R.colors.grey_100,
  },
  gridLine: {
    height: HOUR_HEIGHT,
    borderBottomWidth: 1,
    borderBottomColor: R.colors.grey_100,
    width: '100%',
  },
  timeText: {
    fontSize: R.fontsize.fontsSize12,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.black,
  },
  eventCard: {
    borderRadius: 15,
    paddingLeft: 15,
    paddingTop: 10,
  },
  eventTitle: {
    fontSize: R.fontsize.fontsSize12,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.white,
    fontWeight: '400',
    marginBottom: 5,
  },
  eventSubtitle: {
    fontSize: R.fontsize.fontsSize10,
    fontFamily: R.fonts.InterRegular,
    fontWeight: '400',
    color: R.colors.white,
    marginBottom: 5,
  },
  eventTime: {
    fontSize: R.fontsize.fontsSize10,
    fontFamily: R.fonts.InterRegular,
    color: R.colors.white,
    fontWeight: '400',
  },
});

export default FilterDateView