import React, {useState, useRef, useEffect} from 'react';
import {useNavigation, useFocusEffect} from '@react-navigation/native';
import {DeviceEventEmitter, PanResponder} from 'react-native';
import FilterDayView from './view';

const FilterDay = ({navigation}) => {
  const [currentDate, setCurrentDate] = useState(new Date());
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [showMonthPicker, setShowMonthPicker] = useState(false);
  const scrollViewRef = useRef(null);

  useEffect(() => {
    DeviceEventEmitter.emit('onDateChange', selectedDate);
  }, [selectedDate]);

  // Reset về ngày hiện tại khi chuyển màn hình
  useFocusEffect(
    React.useCallback(() => {
      const today = new Date();
      setCurrentDate(today);
      setSelectedDate(today);
      DeviceEventEmitter.emit('onDateChange', today);
      // Cập nhật header drawer với tháng hiện tại
      DeviceEventEmitter.emit('updateHeaderMonth', today.getMonth());
    }, [])
  );

  const createMockEvents = () => {
    const today = new Date();
    const formatDateToString = date => {
      const year = date.getFullYear();
      const month = (date.getMonth() + 1).toString().padStart(2, '0');
      const day = date.getDate().toString().padStart(2, '0');
      return `${year}-${month}-${day}`;
    };
    
    const todayStr = formatDateToString(today);
    const tomorrow = new Date(today);
    tomorrow.setDate(today.getDate() + 1);
    const tomorrowStr = formatDateToString(tomorrow);
    
    const yesterday = new Date(today);
    yesterday.setDate(today.getDate() - 1);
    const yesterdayStr = formatDateToString(yesterday);

    return [
      // Sự kiện hôm nay
      {
        id: '1',
        title: 'Lịch vào trực lớp TTCĐT 445.T1',
        subtitle: 'CS Địa lý 4D',
        time: '07:00',
        endTime: '09:00',
        date: todayStr,
        type: 'class',
      },
      {
        id: '2',
        title: 'Meeting team development',
        subtitle: 'Phòng họp A1',
        time: '10:00',
        endTime: '11:30',
        date: todayStr,
        type: 'meeting',
      },
      {
        id: '3',
        title: 'Nghỉ giải lao',
        subtitle: 'Thư giãn',
        time: '11:30',
        endTime: '12:30',
        date: todayStr,
        type: 'break',
      },
      {
        id: '3',
        title: 'Đang ngủ',
        subtitle: 'Thư giãn',
        time: '12:30',
        endTime: '13:30',
        date: todayStr,
        type: 'break',
      },
      {
        id: '4',
        title: 'Ăn trưa',
        subtitle: 'Căng tin trường',
        time: '12:00',
        endTime: '13:00',
        date: todayStr,
        type: 'meal',
      },
      {
        id: '5',
        title: 'Training React Native',
        subtitle: 'Online Zoom',
        time: '14:00',
        endTime: '16:00',
        date: todayStr,
        type: 'training',
      },
      {
        id: '6',
        title: 'Code Review Session',
        subtitle: 'Dev Team',
        time: '16:30',
        endTime: '17:30',
        date: todayStr,
        type: 'review',
      },
      
      // Sự kiện ngày mai
      {
        id: '7',
        title: 'Họp nội bộ khoa CNTT',
        subtitle: 'Phòng họp B2',
        time: '08:00',
        endTime: '10:00',
        date: tomorrowStr,
        type: 'meeting',
      },
      {
        id: '8',
        title: 'Lịch học lớp EWC45.364.L1',
        subtitle: 'Tiếng Anh chuyên ngành',
        time: '10:30',
        endTime: '12:00',
        date: tomorrowStr,
        type: 'class',
      },
      {
        id: '9',
        title: 'Workshop AI trong giáo dục',
        subtitle: 'Hội trường lớn',
        time: '13:30',
        endTime: '17:00',
        date: tomorrowStr,
        type: 'workshop',
      },
      
      // Sự kiện hôm qua
      {
        id: '10',
        title: 'Seminar Công nghệ mới',
        subtitle: 'Phòng 301',
        time: '09:00',
        endTime: '11:00',
        date: yesterdayStr,
        type: 'seminar',
      },
      {
        id: '11',
        title: 'Chấm bài thi cuối kỳ',
        subtitle: 'Văn phòng giảng viên',
        time: '14:00',
        endTime: '17:00',
        date: yesterdayStr,
        type: 'grading',
      },
      
      // Sự kiện ngày cố định
      {
        id: '12',
        title: 'Lịch vào trực lớp IT47.8F7',
        subtitle: 'Môn học chuyên ngành',
        time: '07:30',
        endTime: '09:00',
        date: '2025-01-15',
        type: 'class',
      },
      {
        id: '13',
        title: 'Hội nghị khoa học sinh viên',
        subtitle: 'Hội trường A',
        time: '08:30',
        endTime: '14:00',
        date: '2025-01-20',
        type: 'conference',
      },
      {
        id: '14',
        title: 'Bảo vệ đồ án tốt nghiệp',
        subtitle: 'Phòng 205',
        time: '13:00',
        endTime: '17:30',
        date: '2025-01-25',
        type: 'defense',
      },
      {
        id: '15',
        title: 'Tổng kết học kỳ',
        subtitle: 'Phòng họp khoa',
        time: '15:00',
        endTime: '18:00',
        date: '2025-01-30',
        type: 'summary',
      },
    ];
  };

  const mockEvents = createMockEvents();

  const formatDateToString = date => {
    const year = date.getFullYear();
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const day = date.getDate().toString().padStart(2, '0');
    return `${year}-${month}-${day}`;
  };

  const getEventsForDate = date => {
    const dateStr = formatDateToString(date);
    return mockEvents.filter(event => event.date === dateStr);
  };

  const getDayName = date => {
    const days = ['CN', 'T2', 'T3', 'T4', 'T5', 'T6', 'T7'];
    return days[date.getDay()];
  };

  const getMonthName = monthIndex => {
    const months = [
      'Tháng 1',
      'Tháng 2',
      'Tháng 3',
      'Tháng 4',
      'Tháng 5',
      'Tháng 6',
      'Tháng 7',
      'Tháng 8',
      'Tháng 9',
      'Tháng 10',
      'Tháng 11',
      'Tháng 12',
    ];
    return months[monthIndex];
  };

  const handleMonthSelect = monthIndex => {
    const newDate = new Date(currentDate);
    newDate.setMonth(monthIndex);
    setCurrentDate(newDate);
    setSelectedDate(newDate);
    setShowMonthPicker(false);
    DeviceEventEmitter.emit('onDateChange', newDate);
    // Cập nhật header drawer với tháng mới
    DeviceEventEmitter.emit('updateHeaderMonth', monthIndex);
  };

  const swipeToNextDay = () => {
    const nextDay = new Date(selectedDate);
    nextDay.setDate(selectedDate.getDate() + 1);
    setSelectedDate(nextDay);
    setCurrentDate(nextDay);
    DeviceEventEmitter.emit('onDateChange', nextDay);
    // Cập nhật header drawer nếu tháng thay đổi
    DeviceEventEmitter.emit('updateHeaderMonth', nextDay.getMonth());
  };

  const swipeToPrevDay = () => {
    const prevDay = new Date(selectedDate);
    prevDay.setDate(selectedDate.getDate() - 1);
    setSelectedDate(prevDay);
    setCurrentDate(prevDay);
    DeviceEventEmitter.emit('onDateChange', prevDay);
    // Cập nhật header drawer nếu tháng thay đổi
    DeviceEventEmitter.emit('updateHeaderMonth', prevDay.getMonth());
  };

  const toggleMonthPicker = () => {
    setShowMonthPicker(!showMonthPicker);
  };

  const panResponder = PanResponder.create({
    onMoveShouldSetPanResponder: (evt, gestureState) => {
      return Math.abs(gestureState.dx) > 30 && Math.abs(gestureState.dy) < 100;
    },
    onPanResponderMove: (evt, gestureState) => {},
    onPanResponderRelease: (evt, gestureState) => {
      if (gestureState.dx > 50) {
        swipeToPrevDay();
      } else if (gestureState.dx < -50) {
        swipeToNextDay();
      }
    },
  });

  const calculateEventPosition = (startTime, endTime) => {
    const [startHour, startMinute] = startTime.split(':').map(Number);
    const [endHour, endMinute] = endTime.split(':').map(Number);

    const startTotalMinutes = startHour * 60 + startMinute;
    const endTotalMinutes = endHour * 60 + endMinute;
    const durationMinutes = endTotalMinutes - startTotalMinutes;

    const HOUR_HEIGHT = 80;
    const topPosition = (startTotalMinutes / 60) * HOUR_HEIGHT;
    const height = (durationMinutes / 60) * HOUR_HEIGHT;

    return {topPosition, height};
  };

  const calculateEventLayout = (events) => {
    return events.map((event, index) => {
      const {topPosition, height} = calculateEventPosition(event.time, event.endTime);
      
      // Simple stacking - each subsequent event gets slightly offset
      const stackOffset = index * 8; // 8px offset for each event
      const leftOffset = 5 + stackOffset;
      const rightOffset = 15 + stackOffset;
      
      return {
        ...event,
        topPosition,
        height,
        leftOffset,
        rightOffset,
        zIndex: 10 + index // Higher z-index for later events
      };
    });
  };

  return (
    <FilterDayView
      navigation={navigation}
      currentDate={currentDate}
      selectedDate={selectedDate}
      showMonthPicker={showMonthPicker}
      scrollViewRef={scrollViewRef}
      panResponder={panResponder}
      getEventsForDate={getEventsForDate}
      getDayName={getDayName}
      getMonthName={getMonthName}
      handleMonthSelect={handleMonthSelect}
      toggleMonthPicker={toggleMonthPicker}
      calculateEventPosition={calculateEventPosition}
      calculateEventLayout={calculateEventLayout}
    />
  );
};

export default FilterDay;
