import React, { useEffect, useState } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  Image,
  ScrollView,
} from 'react-native';
import R from '../../assets/R';

/**
 * @param items là một mảng [Object1,Object2]
*/

const Dropdown = ({
  items = [],
  placeholder = 'Chọn...',
  onSelect,
  height,
  backgroundColor,
  containerMarginBottom = 5,
  // mới thêm
  editable = true,
  disabledBackgroundColor = R.colors.blue1,
  disabledTextColor = '#999',
  disabledIconColor = '#999',
}) => {
  const [isOpen, setIsOpen] = useState(false);
  const [selected, setSelected] = useState(null);

  const headerHeight = height ?? 35;

  useEffect(() => {
    if (!editable && isOpen) setIsOpen(false);
  }, [editable, isOpen]);

  const handleToggle = () => {
    if (!editable) return;
    setIsOpen(prev => !prev);
  };

  const handleSelect = (item) => {
    if (!editable) return;
    setSelected(item);
    setIsOpen(false);
    onSelect && onSelect(item);
  };

  const resolvedBg = editable ? (backgroundColor || '#fff') : disabledBackgroundColor;
  const resolvedTextColor = editable ? '#333' : disabledTextColor;
  const resolvedIconTint = editable ? '#333' : disabledIconColor;
  const resolvedBorderColor = editable ? R.colors?.grayBorderInputTextHeader || '#ccc' : '#ddd';

  return (
    <View style={[styles.container , {marginBottom:containerMarginBottom}]}>
      {/* Header */}
      <TouchableOpacity
        activeOpacity={editable ? 0.7 : 1}
        onPress={handleToggle}
        disabled={!editable}
        style={[
          styles.dropdownHeader,
          {
            height: headerHeight,
            backgroundColor: resolvedBg,
            borderColor: resolvedBorderColor,
            opacity: editable ? 1 : 0.8,
          },
        ]}
      >
        <Text style={[styles.dropdownHeaderText, { color: resolvedTextColor }]}>
          {selected ? selected.label : placeholder}
        </Text>
        <Image
          source={R.images.icDrop}
          style={[styles.imageIcon, { tintColor: resolvedIconTint }]}
        />
      </TouchableOpacity>

      {/* List */}
      {isOpen && (
        <View
          style={[
            styles.dropdownList,
            {
              top: headerHeight,
              backgroundColor: '#fff',
              borderColor: '#e5e5e5',
            },
          ]}
        >
          <ScrollView style={{ maxHeight: 220 }}>
            {items.map((item) => (
              <TouchableOpacity
                key={String(item.id)}
                style={styles.dropdownItem}
                onPress={() => handleSelect(item)}
              >
                <Text style={styles.dropdownItemText}>{item.label}</Text>
              </TouchableOpacity>
            ))}
          </ScrollView>
        </View>
      )}
    </View>
  );
};

export default Dropdown;

const styles = StyleSheet.create({
  container: {
    width: '100%',
    position: 'relative',
  },
  dropdownHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    borderWidth: 1,
    borderRadius: 5,
    paddingHorizontal: 15,
  },
  dropdownHeaderText: {
    fontSize: 12,
  },
  dropdownList: {
    position: 'absolute',
    left: 0,
    right: 0,
    zIndex: 999,
    elevation: 8,              // Android
    borderWidth: 1,
    borderRadius: 6,
    shadowColor: '#000',       // iOS
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  dropdownItem: {
    paddingVertical: 8,
    paddingHorizontal: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownItemText: {
    fontSize: 12,
    color: '#333',
  },
  imageIcon: {
    width: 20,
    height: 20,
    resizeMode: 'contain',
  },
});
