import React, {useState} from 'react';
import {
  Modal,
  View,
  Text,
  TouchableOpacity,
  ScrollView,
  Image,
  FlatList,
} from 'react-native';
import R from '../../../assets/R';
import styles from './style';
import TextField from '../../../components/Input/TextField';
import TextMulti from '../../../components/Input/TextMulti';
import Dropdown from '../../../components/DropdownAlert/Dropdown';
import Button from '../../../components/Button';

const AddMonitoringModal = ({visible, onClose, onSave}) => {
  const [formData, setFormData] = useState({
    title: '',
    content: '',
    category: null,
    responsible: null,
    deadline: null,
    assignee: null,
    implementer: null,
    attachments: [],
  });

  const [dataList, setDataList] = useState([
    {id: 1, name: 'Nguyễn Minh Đức'},
    {id: 2, name: 'Trần Văn Hùng'},
    {id: 3, name: 'Lê Thị Mai'},
    {id: 4, name: 'Phạm Quốc Khánh'},
    {id: 5, name: 'Hoàng Anh Tuấn'},
    {id: 6, name: 'Vũ Thị Hằng'},
    {id: 7, name: 'Ngô Văn Nam'},
    {id: 8, name: 'Đinh Thị Lan'},
    {id: 9, name: 'Bùi Văn Phúc'},
    {id: 10, name: 'Lý Thị Hoa'},
    {id: 11, name: 'Phan Minh Hoàng'},
    {id: 12, name: 'Tạ Thị Hương'},
    {id: 13, name: 'Đoàn Văn Dũng'},
    {id: 14, name: 'Nguyễn Thị Vân'},
    {id: 15, name: 'Trương Văn Long'},
    {id: 16, name: 'Mai Thị Ngọc'},
    {id: 17, name: 'Huỳnh Quốc Việt'},
    {id: 18, name: 'Lâm Thị Thu'},
    {id: 19, name: 'Nguyễn Hữu Tài'},
    {id: 20, name: 'Phạm Thị Kim'},
  ]);

  const [selectedItems, setSelectedItems] = useState([]);

  const updateFormData = (field, value) => {
    setFormData(prev => ({...prev, [field]: value}));
  };

  const handleSave = () => {
    if (formData.title.trim() && formData.content.trim()) {
      const workData = {
        ...formData,
        selectedImplementers: selectedItems,
        type: 'monitoring', // Đánh dấu đây là công việc giám sát
      };
      onSave(workData);
      resetForm();
    } else {
      alert('Vui lòng điền đầy đủ thông tin bắt buộc');
    }
  };

  const resetForm = () => {
    setFormData({
      title: '',
      content: '',
      category: null,
      responsible: null,
      deadline: null,
      assignee: null,
      implementer: null,
      attachments: [],
    });
    setSelectedItems([]);
  };

  const handleClose = () => {
    resetForm();
    onClose();
  };

  const handleSelectItem = item => {
    const isSelected = selectedItems.some(selected => selected.id === item.id);
    if (isSelected) {
      setSelectedItems(prev => prev.filter(selected => selected.id !== item.id));
    } else {
      setSelectedItems(prev => [...prev, item]);
    }
  };

  const handleRemoveItem = itemId => {
    setSelectedItems(prev => prev.filter(item => item.id !== itemId));
  };

  const renderItem = ({item}) => {
    const isSelected = selectedItems.some(selected => selected.id === item.id);
    return (
      <TouchableOpacity
        style={[styles.chip, isSelected && styles.chipSelected]}
        onPress={() => handleSelectItem(item)}>
        <Text style={[styles.textChip, isSelected && styles.textChipSelected]}>
          {item.name}
        </Text>
      </TouchableOpacity>
    );
  };

  const renderSelectedItem = ({item}) => {
    return (
      <View style={styles.selectedChip}>
        <TouchableOpacity
          style={styles.containerIcon}
          onPress={() => handleRemoveItem(item.id)}>
          <Image
            resizeMode="cover"
            source={R.images.icCancel}
            style={styles.imageIcon}
            tintColor={R.colors.blue}
          />
        </TouchableOpacity>
        <Text style={styles.textChip}>{item.name}</Text>
      </View>
    );
  };

  return (
    <Modal
      visible={visible}
      transparent={true}
      animationType="slide"
      onRequestClose={handleClose}>
      <View style={styles.overlay}>
        <View style={styles.modalContainer}>
          <ScrollView showsVerticalScrollIndicator={false}>
            <Text style={styles.title}>Tạo công việc giám sát</Text>

            <TextField
              title="Tiêu đề công việc"
              placeholder="Nhập tiêu đề công việc"
              required={true}
              value={formData.title}
              onChangeText={value => updateFormData('title', value)}
              containerMarginBottom={15}
            />

            <View style={styles.containerDropdown}>
              <Text style={styles.label}>
                Loại công việc <Text style={styles.required}>*</Text>
              </Text>
              <Dropdown
                height={35}
                placeholder="Chọn loại công việc"
                onSelect={item => updateFormData('category', item)}
              />
            </View>

            <View style={styles.containerDropdown}>
              <Text style={styles.label}>
                Người chịu trách nhiệm <Text style={styles.required}>*</Text>
              </Text>
              <Dropdown
                height={35}
                placeholder="Chọn người chịu trách nhiệm"
                onSelect={item => updateFormData('responsible', item)}
              />
            </View>

            <View style={styles.containerDropdown}>
              <Text style={styles.label}>
                Thời hạn hoàn thành <Text style={styles.required}>*</Text>
              </Text>
              <Dropdown
                height={35}
                placeholder="Chọn thời hạn"
                onSelect={item => updateFormData('deadline', item)}
              />
            </View>

            <TextMulti
              title="Nội dung công việc"
              placeholder="Nhập nội dung chi tiết công việc cần giám sát"
              required={true}
              value={formData.content}
              onChangeText={value => updateFormData('content', value)}
              containerMarginBottom={15}
            />

            <View style={styles.containerDropdown}>
              <Text style={styles.label}>
                Người thực hiện <Text style={styles.required}>*</Text>
              </Text>
              <Dropdown
                height={35}
                placeholder="Chọn người thực hiện"
                onSelect={item => updateFormData('implementer', item)}
              />
            </View>

            <Text style={styles.label}>Danh sách người thực hiện:</Text>
            <FlatList
              data={dataList}
              renderItem={renderItem}
              keyExtractor={item => item.id.toString()}
              numColumns={2}
              style={styles.flatList}
              showsVerticalScrollIndicator={false}
              columnWrapperStyle={styles.flatListRow}
              nestedScrollEnabled={true}
              scrollEnabled={false}
            />

            <Text style={styles.label}>Đã chọn:</Text>
            <FlatList
              data={selectedItems}
              renderItem={renderSelectedItem}
              keyExtractor={item => item.id.toString()}
              numColumns={2}
              style={styles.selectedFlatList}
              showsVerticalScrollIndicator={false}
              columnWrapperStyle={styles.flatListRow}
              nestedScrollEnabled={true}
              scrollEnabled={false}
            />

            <Text style={styles.label}>Tài liệu đính kèm</Text>
            <TouchableOpacity style={styles.attachmentContainer}>
              <Image source={R.images.icDocument} style={styles.uploadIcon} />
              <Text style={styles.attachmentText}>Thêm tài liệu</Text>
            </TouchableOpacity>

            <View style={styles.buttonContainer}>
              <Button
                title="Hủy"
                onPress={handleClose}
                backgroundColor={R.colors.orange}
                width={100}
                height={35}
                borderRadius={100}
                marginRight={10}
                fontSize={R.fontsize.fontSizeContent}
                textColor={R.colors.white}
              />
              <Button
                title="Tạo công việc"
                onPress={handleSave}
                backgroundColor={R.colors.blue}
                width={120}
                height={35}
                borderRadius={100}
                fontSize={R.fontsize.fontSizeContent}
                textColor={R.colors.white}
              />
            </View>
          </ScrollView>
        </View>
      </View>
    </Modal>
  );
};

export default AddMonitoringModal;
