import React, {useState, useEffect} from 'react';
import {
  Text,
  View,
  TouchableOpacity,
  StyleSheet,
  ScrollView,
  FlatList,
  RefreshControl,
  Switch,
  TextInput,
} from 'react-native';
import styles from './style';
import Header from '../../../components/Header/Header';
import R from '../../../assets/R';
import Button from '../../../components/Button';

const ListStudentRollCallView = props => {
  const {refreshing, setRefreshing, studentList, setStudentList} = props;

  const onRefresh = () => {
    setRefreshing(true);
    setTimeout(() => {
      setRefreshing(false);
    }, 1000);
  };

  const toggleAttendance = studentId => {
    setStudentList(prevList =>
      prevList.map(student => {
        if (student.id === studentId) {
          const newIsPresent = !student.isPresent;
          // Auto-fill "nghỉ ko phép" when switching to absent if no custom note
          const newNote =
            !newIsPresent && (!student.note || student.note === '')
              ? 'Nghỉ không phép'
              : newIsPresent
              ? ''
              : student.note;

          return {...student, isPresent: newIsPresent, note: newNote};
        }
        return student;
      }),
    );
  };

  const updateStudentNote = (studentId, note) => {
    setStudentList(prevList =>
      prevList.map(student =>
        student.id === studentId ? {...student, note: note} : student,
      ),
    );
  };

  const countAttendance = () => {
    const present = studentList.filter(
      studentList => studentList.isPresent,
    ).length;
    const absent = studentList.filter(
      studentList => !studentList.isPresent,
    ).length;
    console.log(`Có mặt: ${present}, Vắng mặt: ${absent}`);
    return {present, absent};
  };

  useEffect(() => {
    countAttendance();
  }, [studentList]);

  const renderTableHeader = () => (
    <View style={styles.tableHeader}>
      <View style={[styles.headerCell, styles.codeColumn]}>
        <Text style={styles.headerText}>Mã sv</Text>
      </View>
      <View style={[styles.headerCell, styles.nameColumn]}>
        <Text style={styles.headerText}>Họ và tên</Text>
      </View>
      <View style={[styles.headerCell, styles.dateColumn]}>
        <Text style={styles.headerText}>Ngày học</Text>
      </View>
      <View style={[styles.headerCell, styles.statusColumn]}>
        <Text style={styles.headerText}>Trạng thái</Text>
      </View>
      <View style={[styles.headerCell, styles.noteColumn]}>
        <Text style={styles.headerText}>Ghi chú</Text>
      </View>
    </View>
  );

  const renderStudentItem = ({item}) => (
    <View style={styles.tableRow}>
      <View style={[styles.tableCell, styles.codeColumn]}>
        <Text style={styles.cellText}>{item.studentCode}</Text>
      </View>
      <View style={[styles.tableCell, styles.nameColumn]}>
        <Text style={[styles.cellText, {textAlign: 'left'}]}>{item.name}</Text>
      </View>
      <View style={[styles.tableCell, styles.dateColumn]}>
        <Text style={styles.cellText}>{item.date}</Text>
      </View>
      <View
        style={[styles.tableCell, styles.statusColumn, styles.statusContainer]}>
        <Switch
          value={item.isPresent}
          onValueChange={() => toggleAttendance(item.id)}
          trackColor={{false: R.colors.red, true: R.colors.green}}
          thumbColor={item.isPresent ? R.colors.white : R.colors.white}
        />
        <Text
          style={[
            styles.statusText,
            {color: item.isPresent ? R.colors.green : R.colors.red},
          ]}>
          {item.isPresent ? 'Có mặt' : 'Vắng mặt'}
        </Text>
      </View>
      <View style={[styles.tableCell, styles.noteColumn]}>
        <TextInput
          style={styles.noteInput}
          value={item.note || ''}
          onChangeText={text => updateStudentNote(item.id, text)}
          placeholder="Ghi chú..."
          placeholderTextColor={R.colors.gray2}
          multiline={true}
          textAlignVertical="top"
        />
      </View>
    </View>
  );

  const {present, absent} = countAttendance();

  return (
    <View style={styles.container}>
      <Header title={'Điểm danh lớp: ATTT2024.P1'} isBack />
      <ScrollView showsVerticalScrollIndicator={false} style={styles.body}>
        <Text style={styles.textTitle}>Danh sách lớp</Text>

        <View style={styles.tableContainer}>
          <ScrollView horizontal showsHorizontalScrollIndicator={false}>
            <View style={styles.tableWrapper}>
              {renderTableHeader()}
              <FlatList
                data={studentList}
                renderItem={renderStudentItem}
                keyExtractor={item => item.id.toString()}
                refreshControl={
                  <RefreshControl
                    refreshing={refreshing}
                    onRefresh={onRefresh}
                    colors={[R.colors.blue]}
                    progressViewOffset={50}
                  />
                }
                showsVerticalScrollIndicator={false}
                scrollEventThrottle={400}
                refreshing={refreshing}
              />
            </View>
          </ScrollView>
        </View>

        <View
          style={{
            flexDirection: 'row',
            justifyContent: 'space-between',
            alignItems: 'center',
            marginVertical: 10,
          }}>
          <Text style={styles.textTitleStatus}>
            Có mặt:{' '}
            <Text style={[styles.statusText, {color: R.colors.green}]}>
              {present}
            </Text>
            , Vắng mặt:{' '}
            <Text style={[styles.statusText, {color: R.colors.red}]}>
              {absent}
            </Text>
          </Text>
          <Button
            title="Đồng bộ kết quả "
            onPress={() => {}}
            containerStyle={{
              paddingHorizontal: 5,
              paddingVertical: 5,
              borderRadius: 20,
            }}
            backgroundColor={R.colors.blue}
            textColor={R.colors.white}
            fontSize={12}
          />
        </View>
        <View>
          <Text
            style={[
              styles.textTitleStatus,
              {
                color: R.colors.blue,
                fontWeight: '600',
                fontFamily: R.fonts.fontMedium,
              },
            ]}>
            Giảng viên lưu ý:
          </Text>
          <Text style={styles.textTitleStatus}>
            Điểm danh chỉ được thực hiện trong 15p trước và 15p sau khi đã vào
            tiếp (trong 30p) sau thời gian điểm danh giảng viên sẽ không được
            cập nhật kết quả điểm danh đã lưu trước đo
          </Text>
        </View>
      </ScrollView>
    </View>
  );
};

export default ListStudentRollCallView;
