import React from 'react';
import {View, Text, TouchableOpacity, ScrollView, FlatList} from 'react-native';
import R from '../../../assets/R';
import styles from './style';

const FilterWeekView = ({
  currentDate,
  showMonthPicker,
  scrollViewRef,
  panResponder,
  isToday,
  getDayName,
  getMonthName,
  getWeekDates,
  getEventsForDate,
  calculateEventPosition,
  handleMonthSelect,
}) => {
  const renderMonthPicker = () => {
    if (!showMonthPicker) return null;

    return (
      <View style={styles.monthPickerContainer}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.monthPickerContent}>
          {Array.from({length: 12}, (_, index) => (
            <TouchableOpacity
              key={index}
              style={[
                styles.monthItem,
                currentDate.getMonth() === index && styles.monthItemSelected,
              ]}
              onPress={() => handleMonthSelect(index)}>
              <Text
                style={[
                  styles.monthItemText,
                  currentDate.getMonth() === index &&
                    styles.monthItemTextSelected,
                ]}>
                {getMonthName(index)}
              </Text>
            </TouchableOpacity>
          ))}
        </ScrollView>
      </View>
    );
  };

  const renderWeekHeader = () => {
    const weekDates = getWeekDates(currentDate);

    return (
      <View style={styles.weekHeaderContainer}>
        <View style={styles.timeColumnHeader} />
        {weekDates.map((date, index) => {
          const isCurrentDay = isToday(date);

          return (
            <View key={index} style={styles.dayHeaderCell}>
              <Text
                style={[
                  styles.dayHeaderText,
                  isCurrentDay && styles.dayHeaderTextToday,
                ]}>
                {getDayName(date)}
              </Text>

              <View
                style={
                  isCurrentDay ? styles.dayHeaderNumberContainerToday : {}
                }>
                <Text
                  style={[
                    styles.dayHeaderNumber,
                    isCurrentDay && styles.dayHeaderNumberToday,
                  ]}>
                  {date.getDate()}
                </Text>
              </View>
            </View>
          );
        })}
      </View>
    );
  };

  const renderTimeSlots = () => {
    const hours = Array.from({length: 24}, (_, i) => i);
    const weekDates = getWeekDates(currentDate);

    return (
      <View style={styles.timeSlotsContainer} {...panResponder.panHandlers}>
        <ScrollView
          ref={scrollViewRef}
          showsVerticalScrollIndicator={false}
          contentContainerStyle={styles.scrollContent}>
          <View style={styles.timelineContainer}>
            <View style={styles.timeLabelsColumn}>
              {hours.map(hour => {
                const timeStr = hour.toString().padStart(2, '0') + ':00';
                return (
                  <View key={hour} style={styles.timeSlot}>
                    <Text style={styles.timeText}>{timeStr}</Text>
                  </View>
                );
              })}
            </View>

            <View style={styles.weekGridContainer}>
              {weekDates.map((date, dayIndex) => (
                <View key={dayIndex} style={styles.dayColumn}>
                  {hours.map(hour => (
                    <View key={hour} style={styles.gridCell} />
                  ))}

                  {getEventsForDate(date).map(event => {
                    const {topPosition, height} = calculateEventPosition(
                      event.time,
                      event.endTime,
                    );

                    return (
                      <TouchableOpacity
                        key={event.id}
                        style={[
                          styles.eventCard,
                          {
                            position: 'absolute',
                            top: topPosition,
                            height: Math.max(height, 40),
                            left: 2,
                            right: 2,
                            zIndex: 10,
                            backgroundColor: R.colors.blue,
                          },
                        ]}
                        activeOpacity={0.7}>
                        <Text
                          style={styles.eventTitle}
                          numberOfLines={height > 60 ? 2 : 1}>
                          {event.title}
                        </Text>
                        {height > 40 && (
                          <Text style={styles.eventSubtitle} numberOfLines={1}>
                            {event.subtitle}
                          </Text>
                        )}
                        <Text style={styles.eventTime}>
                          {event.time} - {event.endTime}
                        </Text>
                      </TouchableOpacity>
                    );
                  })}
                </View>
              ))}
            </View>
          </View>
        </ScrollView>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      {renderMonthPicker()}
      {renderWeekHeader()}
      {renderTimeSlots()}
    </View>
  );
};

export default FilterWeekView;
