import React, {useState, useEffect} from 'react';
import {
    View,
    Text,
    StyleSheet,
    Image,
    TouchableOpacity,
    KeyboardAvoidingView,
    Platform,
    Alert,
} from 'react-native';
import InputIcon from '../../components/Input/InputIcon';
import {getFontXD, HEIGHTXD, WIDTHXD} from '../../Config/Functions';
import R from '../../assets/R';
import {useNavigation} from '@react-navigation/native';
import {
    TABNAVIGATOR,
    FORGOTPASSWORD,
    CONFIRMEMAIL,
} from '../../routers/ScreenNames';
import {checkFormatArray, encryptRSAString} from '../../Config/Functions';
import {loginApi} from '../../apis/Functions/users';
import KEY from '../../assets/AsynStorage';
import AsyncStorage from '@react-native-community/async-storage';
import {showLoading, hideLoading} from '../../actions/loadingAction';
import {saveUserToRedux} from '../../actions/users';
import {connect} from 'react-redux';
import messaging from '@react-native-firebase/messaging';
import I18n from '../../helper/i18/i18n';
import AppText from '../../components/AppText';
import {showAlert, TYPE} from '../../components/DropdownAlert';
import TouchID from 'react-native-touch-id';
import {call} from 'redux-saga/effects';
import KeychainService from '../../services/keychainService';
import * as Keychain from 'react-native-keychain';
import EntypoIcon from 'react-native-vector-icons/Entypo';
import Icon from 'react-native-vector-icons/MaterialCommunityIcons';
import DeviceInfo from 'react-native-device-info';

const Login = (props) => {
    const {navigation} = props;

    const [email, setEmail] = useState('');
    const [pass, setPass] = useState('');
    const [biometryType, setBiometryType] = useState(null);
    const [isShowBiometryLogin, setIsShowBiometryLogin] = useState(false);

    const navigate = useNavigation();
    const optionalConfigObject = {
        title: 'Authentication Required', // Android
        color: '#e00606', // Android,
        fallbackLabel: '', // iOS (if empty, then label is hidden)
    };
    const getTokenDevice = async () => {
        let fcmToken = await AsyncStorage.getItem(KEY.FIREBASE);

        if (!fcmToken) {
            fcmToken = await messaging().getToken();
            if (fcmToken) {
                AsyncStorage.setItem(KEY.FIREBASE, fcmToken);
            }
        }
    };
    const getLoginByBiometry = async () => {
        let loginByBiometry = await AsyncStorage.getItem(KEY.IS_LOGIN_BY_BIOMETRY);
        if (loginByBiometry) {
            loginByBiometry = JSON.parse(loginByBiometry)
            setIsShowBiometryLogin(loginByBiometry.isLoginByBiometry);
            loginByBiometry.isLoginByBiometry && Keychain.getSupportedBiometryType({}).then((biometryType) => {
                setBiometryType(biometryType);
            });
        }
    };
    useEffect(() => {
        props.hideLoading();
        getAccount();
        getTokenDevice();
        getLoginByBiometry();
    }, []);

    const getAccount = async () => {
        const jsonValue = await AsyncStorage.getItem(KEY.ACCOUNT);
        const account = JSON.parse(jsonValue);
        if (account) {
            onSubmitLogin(account.email, account.pass);
        }
    };
    const getCredentialInfo = async () => {
        try {
            const appBundleId = await DeviceInfo.getBundleId(); // only used for android as default for service on iOS is bundleId

            // Retrieve the credentials
            const options = {
                service: Platform.OS === 'ios' ? undefined : appBundleId,
                forceAuth: true,
                authenticationPrompt: {
                    title: 'Authentication needed',
                    cancel: 'Cancel',
                },
            };
            const credentials = await Keychain.getGenericPassword(options);
            if (credentials) {
                console.log(
                    'Credentials successfully loaded for user ', credentials,
                );
                onSubmitLogin(credentials.username, credentials.password);

            } else {
                showAlert(
                    TYPE.ERROR,
                    I18n.t('Notification'),
                    I18n.t('HaveNotCredential', {type: biometryType == 'FaceID' ? I18n.t('FaceId') : I18n.t('Fingerprint')}),
                );
                console.log('No credentials stored');
            }
        } catch (error) {
            console.log('Keychain couldn\'t be accessed!', error);
        }
    };
    const onSubmitLogin = async (email, pass) => {
        const titles = [
            I18n.t('Username').toLowerCase(),
            I18n.t('Password').toLowerCase(),
        ];
        const index = checkFormatArray([email, pass]);

        if (index === true) {
            firebase = await AsyncStorage.getItem(KEY.FIREBASE);
            props.showLoading();
            const res = await loginApi({
                email,
                password: encryptRSAString(pass),
                platform: Platform.OS,
                device_token: firebase,
                account_type: 'CUSTOMER',
            });
            props.hideLoading();
            if (res.data) {
                if (res.data.code == 200 && res.data.data) {
                    const jsonValue = JSON.stringify({email, pass});
                    AsyncStorage.setItem(KEY.TOKEN, res.data.data.token);
                    AsyncStorage.setItem(KEY.ACCOUNT, jsonValue);
                    props.saveUserToRedux(res.data.data);
                    navigate.reset({
                        index: 1,
                        routes: [{name: TABNAVIGATOR}],
                    });
                } else {
                    showAlert(TYPE.ERROR, I18n.t('Notification'), res.data.message);
                }
            } else {
                showAlert(
                    TYPE.ERROR,
                    I18n.t('Notification'),
                    I18n.t('Systemmaintenance'),
                );
            }
        } else {
            showAlert(
                TYPE.WARN,
                I18n.t('Notification'),
                I18n.t('Please_fill_in') + titles[index],
            );
        }
    };

    return (
        <View
            style={{
                flex: 1,
                paddingHorizontal: 20,
                paddingTop: 20,
                backgroundColor: 'white',
            }}>
            <InputIcon
                icon={R.images.iconUser3}
                title={'Email'}
                onChangeText={(val) => setEmail(val)}
                value={email}
            />
            <InputIcon
                icon={R.images.iconLock}
                title={'Password'}
                isPassWord={true}
                onChangeText={(val) => setPass(val)}
                value={pass}
            />
            <TouchableOpacity
                onPress={() => navigate.navigate(CONFIRMEMAIL)}
                style={styles.forgotView}>
                <AppText i18nKey={'ForgotPassword'} style={styles.txtTitle}/>
            </TouchableOpacity>

            <View
                style={{
                    marginVertical: 20,
                    alignItems: 'center',
                }}>
                <TouchableOpacity
                    onPress={() => onSubmitLogin(email, pass)}
                    style={styles.wrapLogin}>
                    <AppText i18nKey={'Login'} style={styles.txtLogin}/>
                    <Image source={R.images.iconRight1} style={styles.imgIcon}/>
                </TouchableOpacity>
                {isShowBiometryLogin ?
                    <View style={{flexDirection: 'row', marginTop: WIDTHXD(70)}}>
                        {biometryType == 'FaceID' ?
                            <TouchableOpacity
                                onPress={() => {
                                    getCredentialInfo();
                                }}>
                                <Image source={R.images.iconFaceId} style={[styles.imgIconBiometry, {tintColor: R.colors.main}]}/>
                            </TouchableOpacity>
                            :
                            <TouchableOpacity
                                onPress={() => {
                                    getCredentialInfo();
                                }}>
                                <Image source={R.images.fingerprint} style={styles.imgIconBiometry}/>
                            </TouchableOpacity>

                        }
                    </View>
                    : null}

                <View style={styles.row}>
                    <AppText i18nKey={'Have_account'} style={styles.txtTitle}/>
                    <TouchableOpacity
                        onPress={() => {
                            navigation.navigate('REGISTOR');
                        }}>
                        <AppText i18nKey={'Register'} style={styles.txtRegistor}/>
                    </TouchableOpacity>
                </View>
            </View>
            <View style={{height: 100}}/>
        </View>
    );
};

const styles = StyleSheet.create({
    forgotView: {
        marginVertical: 20,
        alignItems: 'flex-end',
    },
    txtTitle: {
        fontSize: getFontXD(42),
        color: '#929292',
    },
    txtLogin: {
        fontSize: getFontXD(48),
        color: 'white',
        fontWeight: '700',
    },
    wrapLogin: {
        width: WIDTHXD(512),
        height: HEIGHTXD(150),
        backgroundColor: '#1C6AF6',
        borderRadius: 30,
        flexDirection: 'row',
        justifyContent: 'center',
        alignItems: 'center',
    },
    imgIcon: {
        width: WIDTHXD(72),
        height: HEIGHTXD(72),
        marginLeft: 5,
    },
    imgIconBiometry: {
        width: WIDTHXD(120),
        height: WIDTHXD(120),
    },
    row: {
        flexDirection: 'row',
        marginTop: 30,
    },
    txtRegistor: {
        marginLeft: WIDTHXD(10),
        fontSize: getFontXD(42),
        color: '#1473E6',
    },
});

const mapStateToProps = (state) => {
    return {};
};
export default connect(mapStateToProps, {
    showLoading,
    hideLoading,
    saveUserToRedux,
})(Login);
